<?php

namespace ArchiElite\EventsBooking\Forms;

use ArchiElite\EventsBooking\Enums\AccountStatusEnum;
use ArchiElite\EventsBooking\Enums\AccountTypeEnum;
use ArchiElite\EventsBooking\Enums\EventStatusEnum;
use ArchiElite\EventsBooking\Http\Requests\EventRequest;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\Models\Tax;
use Botble\Base\Facades\Html;
use Botble\Base\Forms\FieldOptions\DatePickerFieldOption;
use Botble\Base\Forms\FieldOptions\EditorFieldOption;
use Botble\Base\Forms\FieldOptions\HtmlFieldOption;
use Botble\Base\Forms\FieldOptions\MediaImageFieldOption;
use Botble\Base\Forms\FieldOptions\NumberFieldOption;
use Botble\Base\Forms\FieldOptions\OnOffFieldOption;
use Botble\Base\Forms\FieldOptions\RepeaterFieldOption;
use Botble\Base\Forms\FieldOptions\SelectFieldOption;
use Botble\Base\Forms\FieldOptions\TextareaFieldOption;
use Botble\Base\Forms\FieldOptions\TextFieldOption;
use Botble\Base\Forms\Fields\DatePickerField;
use Botble\Base\Forms\Fields\EditorField;
use Botble\Base\Forms\Fields\HtmlField;
use Botble\Base\Forms\Fields\MediaImageField;
use Botble\Base\Forms\Fields\NumberField;
use Botble\Base\Forms\Fields\OnOffField;
use Botble\Base\Forms\Fields\RepeaterField;
use Botble\Base\Forms\Fields\SelectField;
use Botble\Base\Forms\Fields\TextareaField;
use Botble\Base\Forms\Fields\TextField;
use Botble\Base\Forms\FormAbstract;
use Botble\Location\Fields\Options\SelectLocationFieldOption;
use Botble\Location\Fields\SelectLocationField;
use Illuminate\Support\Str;

class EventForm extends FormAbstract
{
    public function setup(): void
    {
        $accounts = Account::query()
            ->where('type', AccountTypeEnum::ORGANIZER)
            ->where('status', AccountStatusEnum::ACTIVATED)
            ->pluck('name', 'id')
            ->all();

        $speakers = Account::query()
            ->where('type', AccountTypeEnum::SPEAKER)
            ->where('status', AccountStatusEnum::ACTIVATED)
            ->pluck('name', 'id')
            ->all();

        $taxes = Tax::query()->pluck('title', 'id')->all();

        $fields = [
            [
                'type' => 'number',
                'label' => trans('plugins/events-booking::events-booking.event.form.priority'),
                'attributes' => [
                    'name' => 'priority',
                    'value' => 0,
                    'options' => [
                        'class' => 'form-control',
                    ],
                ],
            ],
            [
                'type' => 'customSelect',
                'label' => trans('plugins/events-booking::events-booking.event.form.speaker'),
                'attributes' => [
                    'name' => 'speaker_id',
                    'list' => $speakers,
                    'value' => null,
                    'options' => [
                        'class' => 'form-control select-search-full',
                    ],
                ],
            ],
            [
                'type' => 'text',
                'label' => trans('plugins/events-booking::events-booking.event.form.title'),
                'attributes' => [
                    'name' => 'title',
                    'value' => null,
                    'options' => [
                        'class' => 'form-control',
                    ],
                ],
            ],
            [
                'type' => 'text',
                'label' => trans('plugins/events-booking::events-booking.event.form.subtitle'),
                'attributes' => [
                    'name' => 'subtitle',
                    'value' => null,
                    'options' => [
                        'class' => 'form-control',
                    ],
                ],
            ],
        ];

        $this
            ->model(Event::class)
            ->setValidatorClass(EventRequest::class)
            ->withCustomFields()
            ->columns()
            ->add(
                'name',
                TextField::class,
                TextFieldOption::make()
                    ->label(trans('core/base::forms.name'))
                    ->placeholder(trans('core/base::forms.name_placeholder'))
                    ->required()
                    ->colspan(2)
            )
            ->add(
                'account_id',
                SelectField::class,
                SelectFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.organizer'))
                    ->searchable()
                    ->choices($accounts)
                    ->required()
                    ->colspan(2)
            )
            ->add(
                'speaker_data',
                RepeaterField::class,
                RepeaterFieldOption::make()
                    ->when($this->getModel()->getKey(), function (RepeaterFieldOption $option) {
                        $data = $this->getModel()->speakers->map(function ($speaker) {
                            return [
                                [
                                    'key' => 'priority',
                                    'value' => $speaker->pivot->priority,
                                ],
                                [
                                    'key' => 'speaker_id',
                                    'value' => $speaker->id,
                                ],
                                [
                                    'key' => 'title',
                                    'value' => $speaker->pivot->title,
                                ],
                                [
                                    'key' => 'subtitle',
                                    'value' => $speaker->pivot->subtitle,
                                ],
                            ];
                        })->toArray();

                        $option->value($data);
                    })
                    ->label(trans('plugins/events-booking::events-booking.event.form.speakers'))
                    ->colspan(2)
                    ->fields($fields)
            )
            ->add(
                'price',
                TextField::class,
                TextFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.price'))
                    ->defaultValue(0)
                    ->required()
                    ->colspan(2)
            )
            ->add(
                'unique_id',
                TextField::class,
                TextFieldOption::make()
                    ->when(! $this->model->getKey() || ! $this->model->unique_id, function (TextFieldOption $option) {
                        $option->value(Str::uuid());
                    })
                    ->label(trans('plugins/events-booking::events-booking.event.form.unique_id'))
                    ->colspan(2)
            )
            ->add(
                'short_description',
                TextareaField::class,
                TextareaFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.short_description'))
                    ->colspan(2)
            )
            ->add(
                'description',
                TextareaField::class,
                TextareaFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.description'))
                    ->colspan(2)
            )
            ->add(
                'is_featured',
                OnOffField::class,
                OnOffFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.is_featured'))
            )
            ->add(
                'content',
                EditorField::class,
                EditorFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.content'))
                    ->colspan(2)
            )
            ->add(
                'seats',
                NumberField::class,
                NumberFieldOption::make()
                    ->required()
                    ->label(trans('plugins/events-booking::events-booking.event.form.seats'))
            )
            ->add(
                'booked_seats',
                NumberField::class,
                NumberFieldOption::make()
                    ->defaultValue(0)
                    ->disabled()
                    ->label(trans('plugins/events-booking::events-booking.event.form.booked_seats'))
            )
            ->when(is_plugin_active('location'), function (FormAbstract $form) {
                $form
                    ->add(
                        'wrapper_location',
                        HtmlField::class,
                        HtmlFieldOption::make()
                            ->content('<div class="col-lg-12">')
                    )
                    ->add(
                        'location_data',
                        SelectLocationField::class,
                        SelectLocationFieldOption::make()
                            ->toArray()
                    )
                    ->add(
                        'wrapper_location_closed',
                        HtmlField::class,
                        HtmlFieldOption::make()
                            ->content('</div>')
                    );
            })
            ->add(
                'address',
                TextField::class,
                TextFieldOption::make()
                    ->required()
                    ->label(trans('plugins/events-booking::events-booking.event.form.address'))
                    ->colspan(2)
            )
            ->add(
                'latitude',
                TextField::class,
                TextFieldOption::make()
                    ->placeholder('Ex: 1.462260')
                    ->label(trans('plugins/events-booking::events-booking.event.form.latitude'))
                    ->helperText(trans('plugins/events-booking::events-booking.event.form.location_helper_text', [
                        'subject' => 'Latitude',
                        'link' => Html::link('https://www.latlong.net/convert-address-to-lat-long.html', 'here'),
                    ]))
            )
            ->add(
                'longitude',
                TextField::class,
                TextFieldOption::make()
                    ->placeholder('Ex: 103.812530')
                    ->label(trans('plugins/events-booking::events-booking.event.form.longitude'))
                    ->helperText(trans('plugins/events-booking::events-booking.event.form.location_helper_text', [
                        'subject' => 'Longitude',
                        'link' => Html::link('https://www.latlong.net/convert-address-to-lat-long.html', 'here'),
                    ]))
            )
            ->add(
                'started_at',
                DatePickerField::class,
                DatePickerFieldOption::make()
                    ->addAttribute('data-options', [
                        'enableTime' => true,
                        'dateFormat' => 'Y-m-d H:i',
                    ])
                    ->required()
                    ->label(trans('plugins/events-booking::events-booking.event.form.start_date'))
            )
            ->add(
                'ended_at',
                DatePickerField::class,
                DatePickerFieldOption::make()
                    ->addAttribute('data-options', [
                        'enableTime' => true,
                        'dateFormat' => 'Y-m-d H:i',
                    ])
                    ->defaultValue(null)
                    ->label(trans('plugins/events-booking::events-booking.event.form.end_date'))
            )
            ->add('booking_closed_at', DatePickerField::class, DatePickerFieldOption::make()
                ->withTimePicker()
                ->defaultValue(null)
                ->colspan(2)
                ->label(trans('plugins/events-booking::events-booking.event.form.booking_closed')))
            ->add(
                'status',
                SelectField::class,
                SelectFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.status'))
                    ->choices(EventStatusEnum::labels())
            )
            ->add(
                'image',
                MediaImageField::class,
                MediaImageFieldOption::make()
                    ->label(trans('plugins/events-booking::events-booking.event.form.image'))
            )
            ->add(
                'tax_id',
                SelectField::class,
                SelectFieldOption::make()
                    ->emptyValue(trans('plugins/events-booking::tax.form.none'))
                    ->choices($taxes)
                    ->searchable()
                    ->label(trans('plugins/events-booking::tax.name'))
            )
            ->setBreakFieldPoint('status');
    }
}
