<?php

namespace ArchiElite\EventsBooking\Http\Controllers;

use ArchiElite\EventsBooking\Enums\OrderStatusEnum;
use ArchiElite\EventsBooking\Forms\EventForm;
use ArchiElite\EventsBooking\Http\Requests\EventRequest;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Tables\Events\AttendeeTable;
use ArchiElite\EventsBooking\Tables\Events\OrderTable;
use ArchiElite\EventsBooking\Tables\EventTable;
use Botble\Base\Http\Actions\DeleteResourceAction;
use Botble\Base\Http\Controllers\BaseController;
use Botble\Table\HeaderActions\HeaderAction;
use Illuminate\Support\Arr;

class EventController extends BaseController
{
    public function __construct()
    {
        $this
            ->breadcrumb()
            ->add(trans('plugins/events-booking::events-booking.name'))
            ->add(trans('plugins/events-booking::events-booking.event.name'), route('events-booking.events.index'));
    }

    public function index(EventTable $table)
    {
        $this->pageTitle(trans('plugins/events-booking::events-booking.event.name'));

        return $table->renderTable();
    }

    public function create()
    {
        $this->pageTitle(trans('plugins/events-booking::events-booking.create'));

        return EventForm::create()->renderForm();
    }

    public function store(EventRequest $request)
    {
        $form = EventForm::create()->setRequest($request);

        $form->saving(function (EventForm $form) use ($request) {
            $model = $form->getModel();

            $model->fill(Arr::except($request->validated(), ['speakers']));

            $model->save();

            if ($request->has('speaker_data')) {
                $model->speakers()->sync($this->transformedSpeakersData($request->input('speaker_data', [])));
            }
        });

        return $this
            ->httpResponse()
            ->setPreviousUrl(route('events-booking.events.index'))
            ->setNextUrl(route('events-booking.events.edit', $form->getModel()->getKey()))
            ->setMessage(trans('core/base::notices.create_success_message'));
    }

    public function edit(Event $event)
    {
        $this->pageTitle(trans('core/base::forms.edit_item', ['name' => $event->name]));

        return EventForm::createFromModel($event)->renderForm();
    }

    public function update(Event $event, EventRequest $request)
    {
        EventForm::createFromModel($event)
            ->saving(function (EventForm $form) use ($request) {
                $model = $form->getModel();

                $model->fill(Arr::except($request->validated(), ['speaker_data']));

                $model->save();

                $speakerData = $request->input('speaker_data', []);

                $speakerData = is_array($speakerData) ? $speakerData : json_decode($speakerData, true);

                $model->speakers()->sync($this->transformedSpeakersData($speakerData));
            });

        return $this
            ->httpResponse()
            ->setPreviousUrl(route('events-booking.events.index'))
            ->withUpdatedSuccessMessage();
    }

    public function destroy(Event $event)
    {
        return DeleteResourceAction::make($event);
    }

    public function analytics(string|int $eventId, OrderTable $orderTable, AttendeeTable $attendeeTable)
    {
        $event = Event::query()
            ->withCount(['attendees', 'orders'])
            ->whereKey($eventId)
            ->first();

        if (! $event) {
            abort(404);
        }

        $this->pageTitle(trans('plugins/events-booking::events-booking.event.analytics.view_analytics', ['name' => $event->name]));

        $viewsCount = $event->views;
        $attendeeCount = $event->attendees_count;

        $ordersCompleted = Order::query()
            ->where('event_id', $eventId)
            ->where('status', OrderStatusEnum::COMPLETED)
            ->get();

        $totalAmount = $ordersCompleted->sum('total') ?? 0;

        $ordersCompletedCount = $ordersCompleted
            ->count();

        $ordersPendingCount = Order::query()
            ->where('event_id', $eventId)
            ->where('status', OrderStatusEnum::PENDING)
            ->count();

        $ordersCancelledCount = Order::query()
            ->where('event_id', $eventId)
            ->where('status', OrderStatusEnum::CANCELLED)
            ->count();

        $orderTable = $orderTable
            ->setAjaxUrl(route('events-booking.events.get-orders', $event->getKey()))
            ->setEventId($event->getKey());
        $attendeeTable = $attendeeTable
            ->addHeaderActions([
                HeaderAction::make('import')
                    ->icon('ti ti-upload')
                    ->label(trans('plugins/events-booking::events-booking.import.attendee.title'))
                    ->permission('events-booking.attendees.import')
                    ->route('tools.data-synchronize.import.attendees.index'),
                HeaderAction::make('export')
                    ->icon('ti ti-download')
                    ->label(trans('plugins/events-booking::events-booking.export.attendee.title'))
                    ->permission('events-booking.attendees.export')
                    ->url(route('tools.data-synchronize.export.attendees.index') . '?event_id=' . $event->getKey()),
            ])
            ->setAjaxUrl(route('events-booking.events.get-attendees', $event->getKey()))
            ->setEventId($event->getKey());

        $data = compact(
            'event',
            'viewsCount',
            'attendeeCount',
            'totalAmount',
            'ordersCompletedCount',
            'ordersPendingCount',
            'ordersCancelledCount',
            'orderTable',
            'attendeeTable',
        );

        return view('plugins/events-booking::events.analytics', $data);
    }

    public function getEventOrders(string $eventId, OrderTable $orderTable)
    {
        $orderTable = $orderTable->setEventId($eventId);

        return $orderTable->renderTable();
    }

    public function getEventAttendees(string $eventId, AttendeeTable $attendees)
    {
        $attendees = $attendees->setEventId($eventId);

        return $attendees->renderTable();
    }

    protected function transformedSpeakersData(array|string $data): array
    {
        if (is_string($data)) {
            return [];
        }

        return array_map(function ($item) {
            $result = [];
            foreach ($item as $field) {
                $result[$field['key']] = $field['value'];
            }

            return $result;
        }, $data);
    }
}
