<?php

namespace ArchiElite\EventsBooking\Http\Controllers;

use ArchiElite\EventsBooking\Enums\OrderStatusEnum;
use ArchiElite\EventsBooking\Facades\InvoiceHelper;
use ArchiElite\EventsBooking\Models\Invoice;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Tables\InvoiceTable;
use Botble\Base\Events\BeforeEditContentEvent;
use Botble\Base\Http\Actions\DeleteResourceAction;
use Botble\Base\Http\Controllers\BaseController;
use Botble\Base\Supports\Breadcrumb;
use Illuminate\Http\Request;

class InvoiceController extends BaseController
{
    protected function breadcrumb(): Breadcrumb
    {
        return parent::breadcrumb()
            ->add(trans('plugins/events-booking::invoice.name'), route('events-booking.invoices.index'));
    }

    public function index(InvoiceTable $table)
    {
        $this->pageTitle(trans('plugins/events-booking::invoice.name'));

        return $table->renderTable();
    }

    public function edit(Invoice $invoice, Request $request)
    {
        event(new BeforeEditContentEvent($request, $invoice));

        $this->pageTitle(trans('core/base::forms.edit_item', ['name' => $invoice->code]));

        return view('plugins/events-booking::invoices.edit', compact('invoice'));
    }

    public function destroy(Invoice $invoice)
    {
        return DeleteResourceAction::make($invoice);
    }

    public function getGenerateInvoice(Invoice $invoice, Request $request)
    {
        if ($request->input('type') === 'print') {
            return InvoiceHelper::streamInvoice($invoice);
        }

        return InvoiceHelper::downloadInvoice($invoice);
    }

    public function generateInvoices()
    {
        $orders = Order::query()
            ->where('status', OrderStatusEnum::COMPLETED)
            ->doesntHave('invoice')
            ->get();

        foreach ($orders as $order) {
            InvoiceHelper::store($order);
        }

        $message = trans('plugins/events-booking::invoice.generate_success_message', ['count' => $orders->count()]);

        if ($orders->isEmpty()) {
            $message = trans('plugins/events-booking::invoice.all_invoices_have_already_generated');
        }

        return $this
            ->httpResponse()
            ->setNextUrl(route('events-booking.invoices.index'))
            ->setMessage($message);
    }
}
