<?php

namespace ArchiElite\EventsBooking\Http\Controllers;

use ArchiElite\EventsBooking\Enums\AccountStatusEnum;
use ArchiElite\EventsBooking\Enums\AccountTypeEnum;
use ArchiElite\EventsBooking\Enums\EventStatusEnum;
use ArchiElite\EventsBooking\Enums\TicketStatusEnum;
use ArchiElite\EventsBooking\Events\OrderCreated;
use ArchiElite\EventsBooking\Facades\BookingHelper;
use ArchiElite\EventsBooking\Facades\OrderHelper;
use ArchiElite\EventsBooking\Http\Requests\CalculateBookingAmountRequest;
use ArchiElite\EventsBooking\Http\Requests\CheckoutRequest;
use ArchiElite\EventsBooking\Http\Requests\InitBookingRequest;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Models\Ticket;
use ArchiElite\EventsBooking\Services\CouponService;
use ArchiElite\EventsBooking\Services\OrderService;
use Botble\Base\Facades\BaseHelper;
use Botble\Base\Http\Controllers\BaseController;
use Botble\Base\Supports\Helper;
use Botble\Optimize\Facades\OptimizerHelper;
use Botble\Payment\Enums\PaymentMethodEnum;
use Botble\Payment\Services\Gateways\BankTransferPaymentService;
use Botble\Payment\Services\Gateways\CodPaymentService;
use Botble\Payment\Supports\PaymentHelper;
use Botble\SeoHelper\Facades\SeoHelper;
use Botble\SeoHelper\SeoOpenGraph;
use Botble\Slug\Facades\SlugHelper;
use Botble\Theme\Facades\Theme;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class PublicController extends BaseController
{
    public function getEvent(string $slug, Request $request)
    {
        $slug = SlugHelper::getSlug($slug, SlugHelper::getPrefix(Event::class));

        if (! $slug) {
            abort(404);
        }

        $event = $slug->reference;

        $isPreviewing = $request->input('preview') && (Auth::guard()->check());

        if (! $isPreviewing && $event->status->getValue() !== EventStatusEnum::APPROVED()->getValue()) {
            abort(404);
        }

        SeoHelper::setTitle($event->name)->setDescription(Str::words($event->description, 120));

        $meta = new SeoOpenGraph();

        $meta->setDescription($event->description);
        $meta->setUrl($event->url);
        $meta->setTitle($event->name);
        $meta->setType('article');

        SeoHelper::setSeoOpenGraph($meta);

        Helper::handleViewCount($event, 'viewed_event');

        Theme::breadcrumb()
            ->add(__('Home'), route('public.index'))
            ->add($event->name, $event->url);

        if (function_exists('admin_bar')) {
            admin_bar()->registerLink(__('Edit this event'), route('events-booking.events.edit', $event->getKey()));
        }

        return Theme::scope('events-booking.event', compact('event'), 'plugins/events-booking::themes.event')->render();
    }

    public function getSpeaker(string $slug, Request $request)
    {
        $slug = SlugHelper::getSlug($slug, SlugHelper::getPrefix(Account::class));

        if (! $slug) {
            abort(404);
        }

        $account = $slug->reference;

        $isPreviewing = $request->input('preview') && (Auth::guard()->check());

        if (! $isPreviewing && $account->status->getValue() !== AccountStatusEnum::ACTIVATED || $account->type->getValue() !== AccountTypeEnum::SPEAKER) {
            abort(404);
        }

        $accountDescription = $account->getMetaData('description', true);

        SeoHelper::setTitle($account->name)->setDescription(Str::words($accountDescription, 120));

        $meta = new SeoOpenGraph();

        $meta->setDescription($accountDescription);
        $meta->setUrl($account->url);
        $meta->setTitle($account->name);
        $meta->setType('article');

        SeoHelper::setSeoOpenGraph($meta);

        Theme::breadcrumb()
            ->add(__('Home'), route('public.index'))
            ->add($account->name, $account->url);

        if (function_exists('admin_bar')) {
            admin_bar()->registerLink(__('Edit this speaker'), route('events-booking.accounts.edit', $account->getKey()));
        }

        return Theme::scope('events-booking.speaker', compact('account'), 'plugins/events-booking::themes.speaker')->render();
    }

    public function postBooking(InitBookingRequest $request)
    {
        $event = Event::query()
            ->with(['currency'])
            ->whereKey($request->input('event_id'))
            ->where(function (Builder $query) {
                $query->whereNull('booking_closed_at')
                    ->orWhere('booking_closed_at', '>', Carbon::now());
            })
            ->first();

        if (! $event) {
            abort(404);
        }

        $token = md5(Str::random(40));

        session([
            $token => $request->except(['_token']),
            'checkout_token' => $token,
        ]);

        return $this
            ->httpResponse()
            ->setNextUrl(route('public.booking.form', $token));
    }

    public function getBooking(string $token)
    {
        SeoHelper::setTitle(__('Booking'));

        OptimizerHelper::disable();

        $sessionData = [];
        if (session()->has($token)) {
            $sessionData = session($token);
        }

        if (empty($sessionData)) {
            abort(404);
        }

        Theme::breadcrumb()
            ->add(__('Booking'), route('public.booking'));

        $event = Event::query()
            ->with(['currency'])
            ->whereKey(Arr::get($sessionData, 'event_id'))
            ->where(function (Builder $query) {
                $query->whereNull('booking_closed_at')
                    ->orWhere('booking_closed_at', '>', Carbon::now());
            })
            ->first();

        if (! $event) {
            abort(404);
        }

        $numberSlots = Arr::get($sessionData, 'number_slots', 1);

        $amount = $event ? ($event->price * $numberSlots) : 0;

        $couponAmount = Arr::get($sessionData, 'coupon_amount', 0);

        $couponCode = Arr::get($sessionData, 'coupon_code');

        $subtotal = $amount;

        $tax = $event->tax;

        $taxAmount = $tax ? ($tax->percentage * $subtotal / 100) : 0;

        $total = $subtotal - $couponAmount + $taxAmount;

        return Theme::scope(
            'events-booking.booking',
            compact(
                'event',
                'tax',
                'taxAmount',
                'numberSlots',
                'subtotal',
                'total',
                'token',
                'couponCode',
                'couponAmount',
            ),
            'plugins/events-booking::themes.booking'
        )->render();
    }

    public function postCheckout(CheckoutRequest $request)
    {
        do_action('form_extra_fields_validate', $request);

        $token = $request->input('token');

        if (! session()->has($token)) {
            if (session()->has('booking_transaction_id')) {
                return $this
                    ->httpResponse()
                    ->setNextUrl(route('public.booking.information', session('booking_transaction_id')));
            }

            abort(404);
        }

        if ($request->input('register_account') == 1) {
            $request->validate([
                'email' => 'required|max:60|min:6|email|unique:eb_accounts,email',
            ]);

            $account = Account::query()->create([
                'name' => BaseHelper::clean($request->input('name')),
                'email' => BaseHelper::clean($request->input('email')),
                'phone' => BaseHelper::clean($request->input('phone')),
                'password' => Hash::make($request->input('password')),
            ]);

            Auth::guard('account')->loginUsingId($account->getKey());
        }

        $sessionData = BookingHelper::getCheckoutData();
        $event = Event::query()->findOrFail($request->input('event_id'));
        $slots = Arr::get($sessionData, 'number_slots');

        $subtotal = $event ? ($event->price * $slots) : 0;

        $couponCode = Arr::get($sessionData, 'coupon_code');
        $couponAmount = 0;

        if ($couponCode) {
            $couponService = new CouponService();

            $coupon = $couponService->getCouponByCode($couponCode);

            if ($coupon !== null) {
                $couponAmount = $couponService->getDiscountAmount(
                    $coupon->type->getValue(),
                    $coupon->value,
                    $subtotal
                );

                $coupon->increment('total_used');
            }
        }

        $tax = $event->tax;

        $taxAmount = $tax ? $tax->percentage * ($subtotal - $couponAmount) / 100 : 0;

        $total = ($subtotal - $couponAmount) + $taxAmount;

        $order = Order::query()->where('token', $token)->first();

        if (! $order) {
            $order = new Order();
            $order->token = $token;
            $order->price = $event->price ?? 0;
            $order->subtotal = $subtotal;
            $order->total = $total;
            $order->tax_amount = $taxAmount;
            $order->coupon_code = $couponCode;
            $order->coupon_amount = $couponAmount;
            $order->quantity = $slots;
            $order->note = $request->input('note');
            $order->account_name = $request->input('name');
            $order->account_email = $request->input('email');
            $order->event_id = $event->getKey();
            $order->currency_id = get_application_currency()->getKey();
            $order->code = Str::upper(Str::random(12));
            $order->event_name = $event->name;
            $order->event_ticket_price = $event->price;

            if (Auth::guard('account')->check()) {
                $order->account_id = Auth::guard('account')->user()->getKey();
            }

            $order->save();
        }

        session()->put('booking_transaction_id', $order->code);

        $tickets = $request->input('tickets', []);

        $ticketsData = [];

        $now = Carbon::now();

        foreach ($tickets as $ticket) {
            $ticketsData[] = [
                'order_id' => $order->getKey(),
                'name' => Arr::get($ticket, 'name'),
                'email' => Arr::get($ticket, 'email'),
                'phone' => Arr::get($ticket, 'phone'),
                'note' => Arr::get($ticket, 'note'),
                'code' => Str::upper(Str::random(12)),
                'created_at' => $now,
                'updated_at' => $now,
                'status' => TicketStatusEnum::PENDING,
            ];
        }

        Ticket::query()->insert($ticketsData);

        $request->merge([
            'order_id' => $order->getKey(),
        ]);

        $data = [
            'error' => false,
            'message' => false,
            'amount' => $order->amount,
            'currency' => strtoupper(get_application_currency()->title),
            'type' => $request->input('payment_method'),
            'charge_id' => null,
        ];

        if (is_plugin_active('payment')) {
            session()->put('selected_payment_method', $data['type']);

            $paymentData = apply_filters(PAYMENT_FILTER_PAYMENT_DATA, [], $request);

            switch ($request->input('payment_method')) {
                case PaymentMethodEnum::COD:
                    $codPaymentService = app(CodPaymentService::class);
                    $data['charge_id'] = $codPaymentService->execute($paymentData);
                    $data['message'] = trans('plugins/payment::payment.payment_pending');

                    break;

                case PaymentMethodEnum::BANK_TRANSFER:
                    $bankTransferPaymentService = app(BankTransferPaymentService::class);
                    $data['charge_id'] = $bankTransferPaymentService->execute($paymentData);
                    $data['message'] = trans('plugins/payment::payment.payment_pending');

                    break;

                default:
                    $data = apply_filters(PAYMENT_FILTER_AFTER_POST_CHECKOUT, $data, $request);

                    break;
            }

            if ($checkoutUrl = Arr::get($data, 'checkoutUrl')) {
                return $this
                    ->httpResponse()
                    ->setError($data['error'])
                    ->setNextUrl($checkoutUrl)
                    ->withInput()
                    ->setMessage($data['message']);
            }

            if ($data['error'] || ! $data['charge_id']) {
                return $this
                    ->httpResponse()
                    ->setError()
                    ->setNextUrl(PaymentHelper::getCancelURL())
                    ->withInput()
                    ->setMessage($data['message'] ?: __('Checkout error!'));
            }

            (new OrderService())->processBooking($order->getKey(), $data['charge_id']);

            $redirectUrl = PaymentHelper::getRedirectURL();
        } else {
            $redirectUrl = route('public.booking.information', $order->code);
        }

        $order = Order::query()->find($order->getKey());

        event(new OrderCreated($order));

        OrderHelper::sendOrderConfirmationEmail($order);

        if ($token = $request->input('token')) {
            session()->forget($token);
            session()->forget('checkout_token');
        }

        return $this
            ->httpResponse()
            ->setNextUrl($redirectUrl)
            ->setMessage(__('Booking successfully!'));
    }

    public function checkoutSuccess(string $token)
    {
        $order = Order::query()
            ->with(['tickets', 'event'])
            ->where('token', $token)
            ->first();

        if (! $order) {
            $order = Order::query()->where('code', session('booking_transaction_id'))->first();
        }

        if (! $order) {
            abort(404);
        }

        SeoHelper::setTitle(__('Booking Information'));

        Theme::breadcrumb()
            ->add(__('Booking'), route('public.booking.information', $order));

        session()->forget('booking_transaction_id');

        return Theme::scope(
            'events-booking.booking-information',
            compact('order'),
            'plugins/events-booking::themes.booking-information',
        )
            ->render();
    }

    public function ajaxCalculateBookingAmount(
        CalculateBookingAmountRequest $request,
    ) {
        $slot = $request->input('quantity', 1);
        $event = Event::query()->findOrFail($request->input('event_id'));

        $amount = $event->price * $slot;

        $sessionData = BookingHelper::getCheckoutData();
        $couponCode = Arr::get($sessionData, 'coupon_code');

        $discountAmount = 0;

        if ($couponCode) {
            $couponService = new CouponService();

            $coupon = $couponService->getCouponByCode($couponCode);

            if ($coupon !== null) {
                $discountAmount = $couponService->getDiscountAmount(
                    $coupon->type->getValue(),
                    $coupon->value,
                    $amount
                );
            }

            $sessionData['coupon_amount'] = $discountAmount;
            $sessionData['coupon_code'] = $couponCode;

            BookingHelper::saveCheckoutData([
                'coupon_amount' => $discountAmount,
                'coupon_code' => $couponCode,
            ]);
        }

        $taxAmount = $event->tax ? $event->tax->percentage * ($amount - $discountAmount) / 100 : 0;

        $totalAmount = ($amount - $discountAmount) + $taxAmount;

        return $this->httpResponse()->setData([
            'total_amount' => format_price($totalAmount),
            'amount_raw' => $totalAmount,
            'sub_total' => format_price($amount),
            'tax_amount' => format_price($taxAmount),
            'discount_amount' => format_price($discountAmount),
        ]);
    }

    public function ajaxGetEvents(Request $request)
    {
        $keyword = $request->input('keyword');

        $events = Event::query()
            ->where('name', 'like', '%' . $keyword . '%')
            ->simplePaginate(10);

        return $this
            ->httpResponse()
            ->setData(
                view(
                    'plugins/events-booking::exports.includes.panel-search-data',
                    compact(
                        'events',
                    )
                )->render()
            );
    }

    public function ajaxChooseEvent(string $eventId)
    {
        $event = Event::query()
            ->withCount('attendees')
            ->findOrFail($eventId);

        return $this
            ->httpResponse()
            ->setData([
                'html' => view('plugins/events-booking::exports.includes.event-selected', compact('event'))->render(),
                'attendees_count' => $event->attendees_count,
                'event_id' => $event->getKey(),
            ]);
    }
}
