<?php

namespace ArchiElite\EventsBooking\Importers;

use ArchiElite\EventsBooking\Enums\AccountStatusEnum;
use ArchiElite\EventsBooking\Enums\AccountTypeEnum;
use ArchiElite\EventsBooking\Enums\EventStatusEnum;
use ArchiElite\EventsBooking\Enums\TicketStatusEnum;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\Models\Ticket;
use Botble\Base\Rules\PhoneNumberRule;
use Botble\DataSynchronize\Importer\ImportColumn;
use Botble\DataSynchronize\Importer\Importer;
use Carbon\Carbon;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class AttendeeImporter extends Importer
{
    public function getLabel(): string
    {
        return trans('plugins/events-booking::events-booking.import.attendee.title');
    }

    public function getHeading(): string
    {
        return $this->getLabel();
    }

    public function getDoneMessage(int $count): string
    {
        return trans('plugins/events-booking::events-booking.import.attendee.done_message', ['count' => $count]);
    }

    public function getValidateUrl(): string
    {
        return route('tools.data-synchronize.import.attendees.validate');
    }

    public function getImportUrl(): string
    {
        return route('tools.data-synchronize.import.attendees.store');
    }

    public function getDownloadExampleUrl(): string
    {
        return route('tools.data-synchronize.import.attendees.download-example');
    }

    public function columns(): array
    {
        return [
            ImportColumn::make('event_id')
                ->label('Event ID')
                ->rules(
                    ['required', 'exists:eb_events,id'],
                    trans('plugins/events-booking::events-booking.import.attendee.rules.event_id')
                ),
            ImportColumn::make('name')
                ->rules(
                    ['required', 'string'],
                    trans('plugins/events-booking::events-booking.import.attendee.rules.name')
                ),
            ImportColumn::make('email')
                ->label('Email')
                ->rules(
                    ['required', 'string'],
                    trans('plugins/events-booking::events-booking.import.attendee.rules.email')
                ),
            ImportColumn::make('phone')
                ->nullable()
                ->rules(
                    ['nullable', new PhoneNumberRule()],
                    trans('plugins/events-booking::events-booking.import.attendee.rules.phone')
                ),
            ImportColumn::make('dob')
                ->nullable()
                ->rules(
                    ['nullable', 'date'],
                    trans('plugins/events-booking::events-booking.import.attendee.rules.dob')
                ),
        ];
    }

    public function examples(): array
    {
        $accounts = Account::query()
            ->take(5)
            ->wherePublished()
            ->select(['name', 'email', 'phone', 'dob'])
            ->get();

        $event = Event::query()
            ->where('status', EventStatusEnum::APPROVED)
            ->first();

        $eventId = $event?->getKey() ?? 1;

        if ($accounts->isNotEmpty()) {
            return $accounts->transform(function ($account) use ($eventId) {
                return [
                    'event_id' => $eventId,
                    'name' => $account->name,
                    'email' => $account->email,
                    'phone' => $account->phone,
                    'dob' => $account->dob,
                ];
            })->all();
        }

        return [
            [
                'event_id' => $eventId,
                'name' => 'John Doe',
                'email' => 'johndoe@gmail.com',
                'phone' => '0123456789',
                'dob' => '1990-01-01',
            ],
            [
                'event_id' => $eventId,
                'name' => 'Jane Smith',
                'email' => 'janesmith@gmail.com',
                'phone' => '0987654321',
                'dob' => '1992-02-02',
            ],
            [
                'event_id' => $eventId,
                'name' => 'Alice Johnson',
                'email' => 'alicejohnson@gmail.com',
                'phone' => '0112233445',
                'dob' => '1988-03-03',
            ],
        ];
    }

    public function handle(array $data): int
    {
        $defaultPassword = Hash::make('password');

        $accounts = [];

        foreach ($data as $item) {
            $item['password'] = $defaultPassword;
            $item['type'] = AccountTypeEnum::ATTENDEE;
            $item['status'] = AccountStatusEnum::ACTIVATED;

            $accounts[] = $item;
        }

        foreach ($accounts as $data) {
            $account = Account::query()->where('email', $data['email'])->first();

            if ($account) {
                $account->update(Arr::except($data, ['event_id', 'email', 'password']));
            } else {
                $account = Account::query()->create(Arr::except($data, ['event_id']));
            }

            if ($data['event_id']) {
                $event = Event::query()->find($data['event_id']);

                Ticket::query()->create([
                    'name' => $account->name,
                    'email' => $account->email,
                    'phone' => $account->phoneNumber,
                    'code' => $ticketCode = Str::upper(Str::random(12)),
                    'created_at' => $now = Carbon::now(),
                    'updated_at' => $now,
                    'status' => TicketStatusEnum::PURCHASED,
                ]);

                if (! $event->attendees()->where('attendee_id', $account->getKey())->first()) {
                    $event->attendees()->attach($account->getKey(), [
                        'hash' => $ticketCode,
                    ]);
                }
            }
        }

        return true;
    }
}
