<?php

namespace ArchiElite\EventsBooking\Tables\Events;

use ArchiElite\EventsBooking\Models\Order;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\BulkChanges\CreatedAtBulkChange;
use Botble\Table\Columns\Column;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\FormattedColumn;
use Botble\Table\Columns\LinkableColumn;
use Botble\Table\Columns\StatusColumn;
use Illuminate\Database\Eloquent\Builder;

class OrderTable extends TableAbstract
{
    public string|int $eventId;

    public function setup(): void
    {
        $this
            ->model(Order::class)
            ->addColumns([
                LinkableColumn::make('id')
                    ->urlUsing(function (LinkableColumn $column) {
                        $order = $column->getItem();

                        if (! $order->id) {
                            return null;
                        }

                        return route('events-booking.orders.edit', $order->id);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.id'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return '#' . $item->id;
                    }),
                LinkableColumn::make('account_id')
                    ->urlUsing(function (LinkableColumn $column) {
                        $order = $column->getItem();

                        if (! $order->account_id) {
                            return null;
                        }

                        return route('events-booking.accounts.edit', $order->account_id);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.account_name'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();
                        $item->loadMissing('account');

                        if (! $account = $item->account) {
                            return $item->account_name ?: '-';
                        }

                        return $account->name;
                    }),
                FormattedColumn::make('price')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return format_price($item->price);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.price')),
                Column::make('quantity')
                    ->alignCenter()
                    ->label(trans('plugins/events-booking::events-booking.order.quantity')),
                FormattedColumn::make('total')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return format_price($item->total);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.total')),
                FormattedColumn::make('payment_id')
                    ->searchable(false)
                    ->label(trans('plugins/events-booking::events-booking.order.payment_status'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        if (! is_plugin_active('payment')) {
                            return '&mdash;';
                        }

                        $item = $column->getItem();
                        $item->loadMissing('payment');

                        if (! $payment = $item->payment) {
                            return '-';
                        }

                        return $payment->status->toHtml();
                    }),
                StatusColumn::make(),
                CreatedAtColumn::make(),
            ])
            ->addBulkChanges([
                CreatedAtBulkChange::make(),
            ])
            ->queryUsing(function (Builder $query) {
                $query
                    ->where('event_id', $this->eventId)
                    ->select([
                        'id',
                        'event_id',
                        'account_id',
                        'payment_id',
                        'account_name',
                        'price',
                        'quantity',
                        'total',
                        'subtotal',
                        'status',
                        'created_at',
                    ]);
            });

        $this->type = self::TABLE_TYPE_SIMPLE;
        $this->defaultSortColumn = 0;
        $this->view = $this->simpleTableView();
    }

    public function setEventId(string $eventId): OrderTable
    {
        $this->eventId = $eventId;

        return $this;
    }

    protected function simpleTableView(): string
    {
        return 'plugins/events-booking::tables.simple-table';
    }
}
